<?php

namespace App\Models;

use App\Models\Customer;
use App\Traits\DurationScopeTrait;
use EloquentFilter\Filterable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Sale extends Model
{
    use HasFactory, Filterable, DurationScopeTrait;

    protected $fillable = [
        'tracking',
        'uuid',
        'took_at',
        'order_taker_id',
        'order_type',
        'cart_total_cost',
        'cart_total_items',
        'cart_total_price',
        'items',
        'profit_after_all',
        'payable_after_all',
        'tax',
        'tax_amount',
        'discount_rate',
        'discount_amount',
        'table_id',
        'is_preparing',
        'chef_id',
        'prepared_at',
        'customer_id',
        'ordered_online',
        'biller_id',
        'completed_at',
        'payment_note',
        'progress',
        'staff_note',
        'payment_method_id',
        'note_for_chef',
        'is_cancelled',
        'cancellation_reason',
        'is_discount_in_percentage',
        'pickup_point_id',
        'shipping_address',
        'cash_book_id',
        'tip_amount',
        'guest_room_reservation_id',
        'is_item_based_taxation',
    ];

    /**
     * Setting default route key
     *
     * @return string
     */
    public function getRouteKeyName(): string
    {
        return 'uuid';
    }

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'items' => 'json',
        'tax' => 'json',
        'is_preparing' => 'boolean',
        'is_cancelled' => 'boolean',
        'is_discount_in_percentage' => 'boolean',
        'is_item_based_taxation' => 'boolean',
    ];

    public function serviceTable(): BelongsTo
    {
        return $this->belongsTo(ServiceTable::class, 'table_id');
    }

    public function pickupPoint(): BelongsTo
    {
        return $this->belongsTo(PickupPoint::class, 'pickup_point_id');
    }

    public function guestRoomReservation(): BelongsTo
    {
        return $this->belongsTo(GuestRoomReservation::class, 'guest_room_reservation_id');
    }

    public function scopeOrderForKitchen($query)
    {
        return $query->where('is_cancelled', false)->whereNull('prepared_at');
    }



    public function scopeSubmittedOrder($query, $bypass = false)
    {
        if ($bypass) {
            return $query;
        }
        return $query->where('is_cancelled', false)->whereNull('completed_at');
    }

    public function scopeOrderForBilling($query)
    {
        return $query->where('is_preparing', true)
            ->whereNull('biller_id')
            ->where('is_cancelled', false)
            ->whereNull('completed_at')
            ->whereNotNull('prepared_at')
            ->whereNotNull('chef_id');
    }

    /**
     * Customer info for current sale
     *
     * @return     BelongsTo  The belongs to.
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    public function saleItems(): HasMany
    {
        return $this->hasMany(SaleItem::class, 'sale_id');
    }

    public function taker(): BelongsTo
    {
        return $this->belongsTo(User::class, 'order_taker_id');
    }

    public function paymentMethod(): BelongsTo
    {
        return $this->belongsTo(PaymentMethod::class, 'payment_method_id');
    }

    public function biller(): BelongsTo
    {
        return $this->belongsTo(User::class, 'biller_id');
    }

    public function chef(): BelongsTo
    {
        return $this->belongsTo(User::class, 'chef_id');
    }

    public function getType()
    {
        $types = collect([
            ['title' => __('Dining'), 'key' => 'dining'],
            ['title' => __('Pickup'), 'key' => 'pickup'],
            ['title' => __('Delivery'), 'key' => 'dilivery'],
            ['title' => __('Room Service'), 'key' => 'roomservice'],
        ]);

        $matchedTitle = $types->where('key', $this->order_type)->pluck('title')->first();
        return $matchedTitle;
    }
}
