<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\ReservationStatusStoreRequest;
use App\Http\Requests\ReservationStatusUpdateRequest;
use App\Http\Resources\ReservationStatusResource;
use App\Models\ReservationStatus;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ReservationStatusController extends ApiController
{
    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware('auth:sanctum');
        $this->middleware('demo')->only(['destroy']);
    }

    /**
     * Reservation statuses list
     *
     * @param  \Illuminate\Http\Request  $request
     * @return JsonResponse
     */
    public function index(Request $request): JsonResponse
    {
        $statuses = ReservationStatus::latest()->get();
        return response()->json(ReservationStatusResource::collection($statuses));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\ReservationStatusStoreRequest  $request
     * @return JsonResponse
     */
    public function store(ReservationStatusStoreRequest $request): JsonResponse
    {
        $status = ReservationStatus::create($request->validated());
        \App\Helpers\ActivityLogger::log(__('New reservation status added successfully'));
        return response()->json([
            'message' => __('Data saved successfully'),
            'status' => $status->id,
        ]);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\ReservationStatus  $reservationStatus
     * @return JsonResponse
     */
    public function show(ReservationStatus $reservationStatus): JsonResponse
    {
        return response()->json(new ReservationStatusResource($reservationStatus));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\ReservationStatusUpdateRequest  $request
     * @param  \App\Models\ReservationStatus  $reservationStatus
     * @return JsonResponse
     */
    public function update(ReservationStatusUpdateRequest $request, ReservationStatus $reservationStatus): JsonResponse
    {
        $reservationStatus->update($request->validated());
        \App\Helpers\ActivityLogger::log(__('Reservation status updated successfully'));

        return response()->json([
            'message' => __('Data updated successfully'),
            'status' => $reservationStatus->id,
        ]);
    }

    /**
     * Destroys the given reservation status.
     *
     * @param  \App\Models\ReservationStatus  $reservationStatus
     * @return JsonResponse
     */
    public function destroy(ReservationStatus $reservationStatus): JsonResponse
    {
        if ($reservationStatus->guestRoomReservations->count() < 1) {
            $reservationStatus->delete();
            \App\Helpers\ActivityLogger::log(__('Reservation status removed successfully'));
            return response()->json(['message' => __('Data removed successfully')]);
        }

        return response()->json(['message' => __('Unable to remove reservation status information is being used')], 422);
    }
}
