<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\BookingSourceStoreRequest;
use App\Http\Requests\BookingSourceUpdateRequest;
use App\Http\Resources\BookingSourceResource;
use App\Models\BookingSource;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class BookingSourceController extends ApiController
{
    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware('auth:sanctum');
        $this->middleware('demo')->only(['destroy']);
    }

    /**
     * Booking sources list
     *
     * @param  \Illuminate\Http\Request  $request
     * @return JsonResponse
     */
    public function index(Request $request): JsonResponse
    {
        $sources = BookingSource::latest()->get();
        return response()->json(BookingSourceResource::collection($sources));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\BookingSourceStoreRequest  $request
     * @return JsonResponse
     */
    public function store(BookingSourceStoreRequest $request): JsonResponse
    {
        $source = BookingSource::create($request->validated());
        \App\Helpers\ActivityLogger::log(__('New booking source added successfully'));
        return response()->json([
            'message' => __('Data saved successfully'),
            'source' => $source->id,
        ]);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\BookingSource  $bookingSource
     * @return JsonResponse
     */
    public function show(BookingSource $bookingSource): JsonResponse
    {
        return response()->json(new BookingSourceResource($bookingSource));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\BookingSourceUpdateRequest  $request
     * @param  \App\Models\BookingSource  $bookingSource
     * @return JsonResponse
     */
    public function update(BookingSourceUpdateRequest $request, BookingSource $bookingSource): JsonResponse
    {
        $bookingSource->update($request->validated());
        \App\Helpers\ActivityLogger::log(__('Booking source updated successfully'));

        return response()->json([
            'message' => __('Data updated successfully'),
            'source' => $bookingSource->id,
        ]);
    }

    /**
     * Destroys the given booking source.
     *
     * @param  \App\Models\BookingSource  $bookingSource
     * @return JsonResponse
     */
    public function destroy(BookingSource $bookingSource): JsonResponse
    {
        if ($bookingSource->guestRoomReservations->count() < 1) {
            $bookingSource->delete();
            \App\Helpers\ActivityLogger::log(__('Booking source removed successfully'));
            return response()->json(['message' => __('Data removed successfully')]);
        }

        return response()->json(['message' => __('Unable to remove booking source information is being used')], 422);
    }
}
